-- FILE: 001_initial_schema.sql
-- PURPOSE: Initial SQLite database schema for Astral Rivals offline/local play
-- DATABASE: SQLite (embedded in Flutter client)
-- AI_CONTEXT: This is the first migration for Phase 2. Creates all tables needed for
--             offline gameplay: local player, decks, card definitions, story progress.
--             This schema is intentionally simplified compared to PostgreSQL version.

-- Migration: 001_initial_schema
-- Created: 2025-11-21
-- Phase: 2 (Core Game Loop)

-- ============================================================================
-- 1. LOCAL PLAYER TABLE
-- ============================================================================
-- Stores local player profile (no authentication needed for offline play)

CREATE TABLE IF NOT EXISTS local_player (
    player_id TEXT PRIMARY KEY,
    player_name TEXT NOT NULL,
    created_at TEXT NOT NULL,
    
    -- Story mode progress
    story_chapter_completed INTEGER DEFAULT 0,
    total_story_wins INTEGER DEFAULT 0,
    
    -- Practice mode stats
    practice_wins INTEGER DEFAULT 0,
    practice_losses INTEGER DEFAULT 0,
    
    -- Preferences (JSON blob)
    preferences TEXT
);

-- ============================================================================
-- 2. LOCAL DECKS TABLE
-- ============================================================================
-- Stores player-created decks for offline play

CREATE TABLE IF NOT EXISTS local_decks (
    deck_id TEXT PRIMARY KEY,
    deck_name TEXT NOT NULL,
    is_active INTEGER NOT NULL DEFAULT 0, -- SQLite uses 0/1 for boolean
    
    -- Deck composition (stored as JSON)
    card_ids TEXT NOT NULL, -- JSON array: ["card_warrior_white_001", ...]
    king_card_id TEXT NOT NULL, -- Selected King unit
    castle_card_id TEXT NOT NULL, -- Selected Castle
    relic_card_ids TEXT, -- JSON array of 3 Relic card IDs
    
    -- Metadata
    karma_alignment TEXT, -- 'white', 'black', 'mixed'
    total_cards INTEGER NOT NULL,
    
    -- Timestamps
    created_at TEXT NOT NULL,
    updated_at TEXT NOT NULL,
    
    -- Constraints
    CHECK (total_cards >= 60 AND total_cards <= 100)
);

CREATE INDEX IF NOT EXISTS idx_local_decks_active ON local_decks(is_active);

-- ============================================================================
-- 3. LOCAL CARD DEFINITIONS TABLE
-- ============================================================================
-- Stores all 60 card definitions (3 Major Arcana + 57 Minor Arcana)
-- Loaded from JSON files at app startup

CREATE TABLE IF NOT EXISTS local_card_definitions (
    card_id TEXT PRIMARY KEY,
    card_name TEXT NOT NULL,
    card_type TEXT NOT NULL, -- 'unit', 'spell', 'trap', 'equipment', 'major_arcana'
    karma TEXT NOT NULL, -- 'white', 'black', 'neutral'
    flow_cost INTEGER NOT NULL,
    rarity TEXT NOT NULL, -- 'common', 'uncommon', 'rare', 'epic', 'legendary'
    
    -- Full card data as JSON blob
    card_data TEXT NOT NULL,
    
    -- Flags
    is_major_arcana INTEGER NOT NULL DEFAULT 0,
    is_available INTEGER NOT NULL DEFAULT 1, -- Can be disabled for balance
    
    -- Metadata
    created_at TEXT NOT NULL,
    updated_at TEXT NOT NULL
);

-- Indexes for efficient card queries
CREATE INDEX IF NOT EXISTS idx_local_card_type ON local_card_definitions(card_type);
CREATE INDEX IF NOT EXISTS idx_local_card_karma ON local_card_definitions(karma);
CREATE INDEX IF NOT EXISTS idx_local_card_rarity ON local_card_definitions(rarity);
CREATE INDEX IF NOT EXISTS idx_local_card_available ON local_card_definitions(is_available);

-- ============================================================================
-- 4. STORY MODE PROGRESS TABLE
-- ============================================================================
-- Tracks player progress through story campaign

CREATE TABLE IF NOT EXISTS story_progress (
    chapter_id TEXT PRIMARY KEY,
    chapter_number INTEGER NOT NULL,
    is_completed INTEGER NOT NULL DEFAULT 0,
    
    -- Chapter details
    stars_earned INTEGER DEFAULT 0, -- 0-3 stars per chapter
    best_time_seconds INTEGER,
    
    -- Unlocks (JSON array of card IDs)
    cards_unlocked TEXT,
    
    -- Timestamp
    completed_at TEXT
);

CREATE INDEX IF NOT EXISTS idx_story_chapter ON story_progress(chapter_number);

-- ============================================================================
-- 5. LOCAL MATCH HISTORY TABLE
-- ============================================================================
-- Stores offline match history (AI opponents, practice mode)

CREATE TABLE IF NOT EXISTS local_match_history (
    match_id TEXT PRIMARY KEY,
    
    -- Match details
    opponent_type TEXT NOT NULL, -- 'ai', 'practice'
    opponent_difficulty TEXT, -- 'easy', 'medium', 'hard'
    
    -- Result
    player_won INTEGER NOT NULL, -- 1 = win, 0 = loss
    win_condition TEXT, -- 'king_death', 'castle_destruction', 'deck_out', 'forfeit'
    
    -- Match stats
    total_turns INTEGER,
    duration_seconds INTEGER,
    
    -- Deck used
    deck_id TEXT,
    
    -- Timestamp
    played_at TEXT NOT NULL
);

CREATE INDEX IF NOT EXISTS idx_local_match_history_date ON local_match_history(played_at DESC);

-- ============================================================================
-- 6. CARD COLLECTION TABLE (Optional - for future card unlocking system)
-- ============================================================================
-- Tracks which cards the player has unlocked (if we implement card collection)

CREATE TABLE IF NOT EXISTS card_collection (
    collection_id TEXT PRIMARY KEY,
    card_id TEXT NOT NULL,
    quantity INTEGER NOT NULL DEFAULT 1, -- Number of copies owned
    is_unlocked INTEGER NOT NULL DEFAULT 0,
    
    -- Acquisition
    unlocked_at TEXT,
    unlock_source TEXT, -- 'story', 'starter_pack', 'purchase', etc.
    
    -- Constraints
    UNIQUE(card_id)
);

CREATE INDEX IF NOT EXISTS idx_card_collection_unlocked ON card_collection(is_unlocked);

-- ============================================================================
-- END OF MIGRATION 001
-- ============================================================================
