-- FILE: 004_fix_deck_schema.sql
-- PURPOSE: Fix local_decks table schema to match database_helper.dart saveDeck method
-- DATABASE: SQLite (embedded in Flutter client)
-- AI_CONTEXT: Migration fixes schema mismatches between 001_initial_schema.sql and
--             the actual saveDeck method in database_helper.dart. The code expects:
--             - deck_id as INTEGER AUTOINCREMENT (not TEXT)
--             - deck_contents column (not card_ids)
--             - is_valid column
--             SQLite doesn't allow easy column renames, so we recreate the table.

-- Migration: 004_fix_deck_schema
-- Created: 2025-12-30
-- Phase: 2 (Core Game Loop)

-- ============================================================================
-- RECREATE LOCAL_DECKS TABLE WITH CORRECT SCHEMA
-- ============================================================================

-- Step 1: Create new table with correct schema
CREATE TABLE IF NOT EXISTS local_decks_new (
    deck_id INTEGER PRIMARY KEY AUTOINCREMENT,
    deck_name TEXT NOT NULL,
    is_active INTEGER NOT NULL DEFAULT 0,
    
    -- Champion card (can be Jack, Queen, or King rank)
    king_card_id TEXT NOT NULL,
    
    -- Selected castle card
    castle_card_id TEXT NOT NULL DEFAULT '',
    
    -- Deck composition (stored as JSON)
    deck_contents TEXT NOT NULL,
    
    -- Socket slots (JSON array)
    socket_slots TEXT,
    
    -- Metadata
    karma_alignment TEXT,
    total_cards INTEGER NOT NULL,
    is_valid INTEGER NOT NULL DEFAULT 0,
    
    -- Timestamps
    created_at TEXT NOT NULL,
    updated_at TEXT NOT NULL,
    
    -- Constraints
    CHECK (total_cards >= 0)
);

-- Step 2: Copy any existing data (mapping old column names to new)
-- Note: card_ids -> deck_contents, and we handle missing columns gracefully
INSERT INTO local_decks_new (deck_name, is_active, king_card_id, castle_card_id, deck_contents, socket_slots, karma_alignment, total_cards, is_valid, created_at, updated_at)
SELECT 
    deck_name,
    is_active,
    king_card_id,
    COALESCE(castle_card_id, ''),
    COALESCE(card_ids, '[]'),
    socket_slots,
    karma_alignment,
    total_cards,
    CASE WHEN total_cards >= 60 AND total_cards <= 100 THEN 1 ELSE 0 END,
    created_at,
    updated_at
FROM local_decks
WHERE deck_name IS NOT NULL;

-- Step 3: Drop old table
DROP TABLE local_decks;

-- Step 4: Rename new table to old name
ALTER TABLE local_decks_new RENAME TO local_decks;

-- Step 5: Recreate index
CREATE INDEX IF NOT EXISTS idx_local_decks_active ON local_decks(is_active);

-- ============================================================================
-- END OF MIGRATION 004
-- ============================================================================
